##################################################################################################
###                                 Supplementary material to                                  ###
### A new confidence interval based on the theory of U-statistics for the area under the curve ###
###                 by J. Kampf, L. Vogel, I. Dykun, T. Rassaf and A. Mahabadi                 ###
##################################################################################################


### Here we implement the different confidence intervals compared in the
### article, i.e. the two new confidence intervals (the one based on Corollary 2
### and the one based on Corollary 3) and the classical confidence intervals
### (DeLong's interval, the Mann-Whitney interval due to Sen and the modified
### Wald interval)

### All functions defined in the sequel have three arguments:
### response  - A vector all whose entries are 0 or 1. Represents the true
###             classification I. 
### predictor - A vector of same length as response all whose entries are in 
###             [0,1]. Represents the predicted value Y^. Ordered in such a way
###             that values close to 0 indicate a classification of 0 and values
###             close to 1 indicate a classification of 1.
### level     - The level of the confidence interval.
### All functions defined in the sequel return a vector of two numbers giving
### the endpoints of the confidence interval.


library(pROC)


AUC_new2<- function(response, predictor, level=0.95){
  n=length(response)
  A=auc(response, predictor, direction="<")
  Sigma = array(0, dim=c(3,3))
  for(i in (1:n)){
    w = c( sum( ((predictor[i]<predictor[-i])+(predictor[i]==predictor[-i])/2) * 
                 (1-response[i]) * response[-i] +
                  ((predictor[-i]<predictor[i])+(predictor[-i]==predictor[i])/2) * 
                  (1-response[-i]) * response[i]),
          (n-1) * (1-response[i])+sum(1-response[-i]), 
          sum(response[-i]) +(n-1)* (response[i]) )
    Sigma= Sigma + w %*% t(w)
    Sigma[1,1] = Sigma[1,1] - sum( (((predictor[i]<predictor[-i])+(predictor[i]==predictor[-i])/2) * 
                                     (1-response[i]) * response[-i] + 
                                    ((predictor[-i]<predictor[i])+(predictor[-i]==predictor[i])/2) * 
                                    (1-response[-i]) * response[i])^2 )
    Sigma[1,2] = Sigma[1,2] - sum( (((predictor[i]<predictor[-i])+(predictor[i]==predictor[-i])/2) * 
                                      (1-response[i]) * response[-i] + 
                                    ((predictor[-i]<predictor[i])+(predictor[-i]==predictor[i])/2) * 
                                      (1-response[-i]) * response[i]) *
                                     (1-response[i]+1-response[-i]))
    Sigma[1,3] = Sigma[1,3] - sum( (((predictor[i]<predictor[-i])+(predictor[i]==predictor[-i])/2) * 
                                      (1-response[i]) * response[-i] + 
                                    ((predictor[-i]<predictor[i])+(predictor[-i]==predictor[i])/2) * 
                                      (1-response[-i]) * response[i]) *
                                     (response[i]+response[-i])  )
    Sigma[2,1] = Sigma[2,1] - sum( (((predictor[i]<predictor[-i])+(predictor[i]==predictor[-i])/2) * 
                                      (1-response[i]) * response[-i] + 
                                    ((predictor[-i]<predictor[i])+(predictor[-i]==predictor[i])/2) * 
                                      (1-response[-i]) * response[i]) *
                                     (1-response[i]+1-response[-i])  )
    Sigma[2,2] = Sigma[2,2] - sum((1-response[i]+1-response[-i])^2)
    Sigma[2,3] = Sigma[2,3] - sum((1-response[i]+1-response[-i])*(response[i]+response[-i]))
    Sigma[3,1] = Sigma[3,1] - sum( (((predictor[i]<predictor[-i])+(predictor[i]==predictor[-i])/2) * 
                                      (1-response[i]) * response[-i] + 
                                    ((predictor[-i]<predictor[i])+(predictor[-i]==predictor[i])/2) * 
                                      (1-response[-i]) * response[i]) *
                                     (response[i]+response[-i])  )
    Sigma[3,2] = Sigma[3,2] - sum((1-response[i]+1-response[-i])*(response[i]+response[-i]))
    Sigma[3,3] = Sigma[3,3] - sum((response[i]+response[-i])^2)
  }
  Sigma = Sigma/(n * (n-1) * (n-2))
  p=0
  for(i in (1:n)){
        p = p + sum(((predictor[i]<predictor[-i])+(predictor[i]==predictor[-i])/2)*
          (1-response[i])*response[-i])
  }
  p = p/(n*(n-1))
  E = c(p, mean(1-response), mean(response))
  Sigma=Sigma- 4*E %*% t(E)
  v = c( n^2/(sum(1-response)* sum(response)),
         -n^3*p/(sum(1-response)^2* sum(response)),
         -n^3*p/(sum(1-response)* sum(response)^2))
  S2= max(c(t(v) %*% Sigma %*% v,0))
  return( c( A+qnorm((1-level)/2)*sqrt(S2/n), A+qnorm(level/2+1/2)*sqrt(S2/n)) );
}



AUC_new3 <- function(response, predictor, level=0.95){
  n=length(response)
  A=auc(response, predictor, direction="<")
  Sigma = array(0, dim=c(3,3))
  for(i in (1:n)){
    w = c( sum( ((predictor[i]<predictor[-i])+(predictor[i]==predictor[-i])/2) * 
                  (1-response[i]) * response[-i] +
                  ((predictor[-i]<predictor[i])+(predictor[-i]==predictor[i])/2) * 
                  (1-response[-i]) * response[i]),
           (n-1) * (1-response[i])+sum(1-response[-i]), 
           sum(response[-i]) +(n-1)* (response[i]) )
    Sigma= Sigma + w %*% t(w)
    Sigma[1,1] = Sigma[1,1] - sum( (((predictor[i]<predictor[-i])+(predictor[i]==predictor[-i])/2) * 
                                      (1-response[i]) * response[-i] + 
                                      ((predictor[-i]<predictor[i])+(predictor[-i]==predictor[i])/2) * 
                                      (1-response[-i]) * response[i])^2 )
    Sigma[1,2] = Sigma[1,2] - sum( (((predictor[i]<predictor[-i])+(predictor[i]==predictor[-i])/2) * 
                                      (1-response[i]) * response[-i] + 
                                      ((predictor[-i]<predictor[i])+(predictor[-i]==predictor[i])/2) * 
                                      (1-response[-i]) * response[i]) *
                                     (1-response[i]+1-response[-i]))
    Sigma[1,3] = Sigma[1,3] - sum( (((predictor[i]<predictor[-i])+(predictor[i]==predictor[-i])/2) * 
                                      (1-response[i]) * response[-i] + 
                                      ((predictor[-i]<predictor[i])+(predictor[-i]==predictor[i])/2) * 
                                      (1-response[-i]) * response[i]) *
                                     (response[i]+response[-i])  )
    Sigma[2,1] = Sigma[2,1] - sum( (((predictor[i]<predictor[-i])+(predictor[i]==predictor[-i])/2) * 
                                      (1-response[i]) * response[-i] + 
                                      ((predictor[-i]<predictor[i])+(predictor[-i]==predictor[i])/2) * 
                                      (1-response[-i]) * response[i]) *
                                     (1-response[i]+1-response[-i])  )
    Sigma[2,2] = Sigma[2,2] - sum((1-response[i]+1-response[-i])^2)
    Sigma[2,3] = Sigma[2,3] - sum((1-response[i]+1-response[-i])*(response[i]+response[-i]))
    Sigma[3,1] = Sigma[3,1] - sum( (((predictor[i]<predictor[-i])+(predictor[i]==predictor[-i])/2) * 
                                      (1-response[i]) * response[-i] + 
                                      ((predictor[-i]<predictor[i])+(predictor[-i]==predictor[i])/2) * 
                                      (1-response[-i]) * response[i]) *
                                     (response[i]+response[-i])  )
    Sigma[3,2] = Sigma[3,2] - sum((1-response[i]+1-response[-i])*(response[i]+response[-i]))
    Sigma[3,3] = Sigma[3,3] - sum((response[i]+response[-i])^2)
  }
  Sigma = Sigma/(n * (n-1) * (n-2))
  p=0
  for(i in (1:n)){
    p = p + sum(((predictor[i]<predictor[-i])+(predictor[i]==predictor[-i])/2)*
                  (1-response[i])*response[-i])
  }
  p = p/(n*(n-1))
  E = c(p, mean(1-response), mean(response))
  Sigma=Sigma- 4*E %*% t(E)
  v = c( n^2/(sum(1-response)* sum(response)),
         -n^3*p/(sum(1-response)^2* sum(response)),
         -n^3*p/(sum(1-response)* sum(response)^2))
  S2= max(c(t(v) %*% Sigma %*% v,0))
  if((A>0)&(A<1)){
  lA = log(A/(1-A)) # logit-function applied to the AUC
  ldA = 1/(A*(1-A)) # derivative of the logit-function applied to the AUC
  d = lA + qnorm((1-level)/2) * ldA * sqrt(S2/n)
  u = lA + qnorm(level/2+1/2) * ldA * sqrt(S2/n)
  Re = c( exp(d)/(exp(d)+1), exp(u)/(exp(u)+1) )
  }
  if(A==0){
    Re = c(0,0);
  }
  if(A==1){
    Re = c(1,1);
  }
  return( Re );
}



AUC_DeLong <- function(response, predictor, level=0.95){
  A=auc(response, predictor, direction="<")
  X_1=predictor[!response]
  Y_1=predictor[as.logical(response)]
  V10=rep(0,length(X_1))
  for(i in (1:length(X_1))){
    V10[i] = mean(Y_1>X_1[i])+mean(Y_1==X_1[i])/2
  }
  V01=rep(0, length(Y_1))
  for(i in (1:length(Y_1))){
    V01[i] = mean(Y_1[i]>X_1) + mean(Y_1[i]==X_1)/2
  }
  
  S10 = 1/(length(X_1)-1) * sum((V10-A)^2)
  S01 = 1/(length(Y_1)-1) * sum((V01-A)^2)
  S = 1/(length(X_1)) * S10 + 1/(length(Y_1)) * S01
  return( c(A+qnorm((1-level)/2)*sqrt(S), A+qnorm(level/2+1/2)*sqrt(S)) )
}


AUC_MWald <- function(response, predictor, level=0.95){
  A=auc(response, predictor, direction="<")
  n=length(response)
  return( c( A+qnorm((1-level)/2)*sqrt(A*(1-A)/(0.75*n-1)), 
             A+qnorm(1/2+level/2)*sqrt(A*(1-A)/(0.75*n-1)) ))
  }


AUC_Mann_Whitney <- function(response, predictor, level=0.95){
  A=auc(response, predictor, direction="<")
  X_1=predictor[!response]
  Y_1=predictor[as.logical(response)]
  V10=rep(0,length(X_1))
  for(i in (1:length(X_1))){
    V10[i] = mean(Y_1<X_1[i])+mean(Y_1==X_1[i])/2
  }
  V01=rep(0, length(Y_1))
  for(i in (1:length(Y_1))){
    V01[i] = mean(Y_1[i]>X_1) + mean(Y_1[i]==X_1)/2
  }
  
  S10 = 1/(length(X_1)-1) * sum((V10-mean(V10))^2)
  S01 = 1/(length(Y_1)-1) * sum((V01-mean(V01))^2)
  S = 1/(length(X_1)) * S10 + 1/(length(Y_1)) * S01
  return( c(A+qnorm((1-level)/2)*sqrt(S), A+qnorm(level/2+1/2)*sqrt(S)) )
}

